<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Outbox_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->helper('leads_helper');
        $this->load->helper('staff_helper');
    }

    /**
     * Save email in outbox tables before sending
     */
    public function save_outbox_email($to, $cc, $bcc, $subject, $content, $files, $lead_id, $lead_status, $email_page, $replay_to, $delivery_status, $mandrill_id, $is_visible = 0)
    {
        $staffid = get_staff_user_id();
        // Insert into tbloutbox_email
        $this->db->insert(db_prefix() . 'outbox_email', [
            'lead_id' => $lead_id,
            'lead_status' => $lead_status,
            'staffid' => $staffid,
            'mandrill_id' => $mandrill_id,
            'is_visible' => $is_visible,
            'delivery_status' => $delivery_status,
            'email_page' => $email_page,
            'send_date' => date('Y-m-d H:i:s'),
            'created_at' => date('Y-m-d H:i:s')
        ]);
        $outbox_id = $this->db->insert_id();

        // Insert content into tbloutbox_email_context
        $this->db->insert(db_prefix() . 'outbox_email_context', [
            'outbox_id' => $outbox_id,
            'email_content' => $content,
            'email_subject' => $subject
        ]);

        // Insert recipients into tbloutbox_to
        $this->insert_recipients($outbox_id, $to, 'to');
        if(!empty($cc)) {
            $this->insert_recipients($outbox_id, $cc, 'cc');
        }
        if(!empty($bcc)) {
            $this->insert_recipients($outbox_id, $bcc, 'bcc');
        }

        $this->insert_replay_to($outbox_id, $replay_to);

        // Insert files into tbloutbox_files
        if (!empty($files)) {
            foreach ($files as $file) {
                $file_type_string = $file['filetype'];
                $first_type_part = explode('/', $file_type_string)[1];
                $this->db->insert(db_prefix() . 'outbox_files', [
                    'outbox_id' => $outbox_id,
                    'file_name' => $file['file_name'],
                    'file_type' => $first_type_part,
                    'file_url' => $file['file_url'],
                    'download_file_url' => $file['download_file_url'],
                    'created_at' => date('Y-m-d H:i:s')
                ]);
            }
        }

        return $outbox_id;
    }

    private function insert_recipients($outbox_id, $emails, $type, $replay_to = null)
    {
        if (is_string($emails)) {
            $emails = array_map('trim', explode(',', $emails));
            $emails = array_filter($emails);
        }
        
        if (!is_array($emails)) {
            $emails = [$emails];
        }

        foreach ($emails as $email) {
            $to_user_role_type = get_client_staff_type_using_email($this, $email);
            $this->db->insert(db_prefix() . 'outbox_to', [
                'outbox_id' => $outbox_id,
                'firstname' => $to_user_role_type['firstname'] ?? '',
                'middlename' => $to_user_role_type['middlename'] ?? '',
                'lastname' => $to_user_role_type['lastname'] ?? '',
                'to_id' => $to_user_role_type['id'],
                'email' => $email,
                'to_type' => $type,
                'to_user_type' => $to_user_role_type['role_type'],
                'replay_to' => $replay_to,
                'created_at' => date('Y-m-d H:i:s')
            ]);
        }
    }

    private function insert_replay_to($outbox_id, $replay_to)
    {
        if (!empty($replay_to)) {
            if (!is_array($replay_to)) {
                $replay_to = [$replay_to];
            }

            foreach ($replay_to as $replayEmails) {
                $this->db->insert(db_prefix() . 'outbox_replay_to', [
                    'outbox_id' => $outbox_id,
                    'email' => $replayEmails['email'],
                    'created_at' => date('Y-m-d H:i:s')
                ]);
            }
        }
    }

    /**
     * Update Delivery Status After Sending
     */
    public function update_delivery_status($outbox_id, $status)
    {
        $this->db->where('id', $outbox_id);
        $this->db->update(db_prefix() . 'outbox_email', [
            'delivery_status' => $status
        ]);
    }
    
    /**
     * Get Outbox Email Details
     * This method Get all related records in the outbox tables based on the outbox ID
    */
    public function get_all_outbox($lead_id, $filters = [], $search = '', $limit = 10, $offset = 0, $get_total = false)
    {
        $table = db_prefix() . 'outbox_email';
        $leadTable = db_prefix() . 'leads';
        $statusTable = db_prefix() . 'leads_status';

        // Step 1: Get total matching rows
        $this->db->from($table . ' as o');
        $this->db->join($leadTable . ' as l', 'l.id = o.lead_id', 'left');
        $this->db->join($statusTable . ' as s', 's.id = l.status', 'left');
        $this->_apply_outbox_filters($lead_id, $filters, $search);
        $total_rows = $this->db->count_all_results();

        if ($get_total) {
            return $total_rows;
        }

        // Step 2: Fetch paginated data
        $this->db->select('o.*, s.name as lead_status_name, s.color as lead_status_color');
        $this->db->from($table . ' as o');
        $this->db->join($leadTable . ' as l', 'l.id = o.lead_id', 'left');
        $this->db->join($statusTable . ' as s', 's.id = o.lead_status', 'left');
        $this->_apply_outbox_filters($lead_id, $filters, $search);
        $this->db->limit($limit, $offset);
        $this->db->order_by('o.id', 'DESC');
        $results = $this->db->get()->result_array();

        if (empty($results)) {
            return [
                'success' => true,
                'data' => [],
                'total_pages' => 0,
                'total_rows' => 0
            ];
        }

        // Step 3: Fetch related data
        $outbox_ids = array_column($results, 'id');
        $to_emails   = $this->get_outbox_to_bulk($outbox_ids);
        $reply_tos   = $this->get_outbox_reply_to_bulk($outbox_ids);
        $contexts    = $this->get_outbox_context_bulk($outbox_ids);
        $attachments = $this->get_outbox_files_bulk($outbox_ids);

        foreach ($results as &$row) {
            $oid = $row['id'];
            $row['to_emails']   = $to_emails[$oid]   ?? [];
            $row['to_names']  =   $to_emails[$oid]['combined_names'] ?? '';
            $row['reply_to']    = $reply_tos[$oid]   ?? [];
            $row['context']     = $contexts[$oid]    ?? [];
            $row['attachments'] = $attachments[$oid] ?? [];

            // Optional formatting
            if ((int)$row['lead_status'] === 10) {
                $row['lead_status'] = [
                    'name'  => 'Ledger',
                    'color' => 'Grey'
                ];
            } else {
                $row['lead_status'] = [
                    'name'  => $row['lead_status_name'],
                    'color' => $row['lead_status_color']
                ];
            }
            unset($row['lead_status_name'], $row['lead_status_color']);
        }

        return [
            'success' => true,
            'data' => $results,
            'total_pages' => ceil($total_rows / $limit),
            'total_rows' => $total_rows
        ];
    }

    // Helper to apply filters
    private function _apply_outbox_filters($lead_id, $filters, $search)
    {
        $outboxToTable = db_prefix() . 'outbox_to';


        $this->db->where('o.lead_id', $lead_id);
        $this->db->where('o.is_visible', 1);

        if (!empty($filters['lead_status'])) {
            $this->db->where_in('o.lead_status', $filters['lead_status']);
        }

        if (!empty($filters['delivery_status'])) {
            $this->db->where_in('o.delivery_status', $filters['delivery_status']);
        }

        if (!empty($search)) {
            $this->db->join($outboxToTable . ' as ot', 'ot.outbox_id = o.id', 'left');
            $this->db->group_start()
                ->like('o.subject', $search)
                ->or_like('o.content', $search)
                ->or_like('ot.firstname', $search)
                ->or_like('ot.middlename', $search)
                ->or_like('ot.lastname', $search)
            ->group_end();
        }
    }

     
    /**
     * Helper Functions (Modular)
    */
    private function get_outbox_to($outbox_id)
    {
        return $this->db->where('outbox_id', $outbox_id)->get(db_prefix() . 'outbox_to')->result_array();
    }

    private function get_outbox_reply_to($outbox_id)
    {
        return $this->db->where('outbox_id', $outbox_id)->get(db_prefix() . 'outbox_replay_to')->result_array();
    }

    private function get_outbox_context($outbox_id)
    {
        return $this->db->where('outbox_id', $outbox_id)->get(db_prefix() . 'outbox_email_context')->result_array();
    }

    private function get_outbox_files($outbox_id)
    {
        return $this->db->where('outbox_id', $outbox_id)->get(db_prefix() . 'outbox_files')->result_array();
    }

    
    /**
     * Bulk Helper Functions
    */
    private function get_outbox_to_bulk($outbox_ids)
    {
        $data = [];
        $rows = $this->db->where_in('outbox_id', $outbox_ids)->get(db_prefix() . 'outbox_to')->result_array();

        foreach ($rows as $row) {
            $user = get_client_staff_data_using_type_id($this, $row['to_id'], $row['to_user_type']);

            $full_name = trim("{$user['firstname']} {$user['middlename']} {$user['lastname']}");
            $row['firstname'] = $user['firstname'] ?? '';
            $row['middlename'] = $user['middlename'] ?? '';
            $row['lastname'] = $user['lastname'] ?? '';
            $row['full_name'] = $full_name;

            $outbox_id = $row['outbox_id'];

            $data[$outbox_id]['recipients'][] = $row;
            $data[$outbox_id]['combined_names'][] = $full_name;
        }

        // Flatten combined_names to string
        foreach ($data as &$item) {
            $item['combined_names'] = implode(', ', array_filter($item['combined_names']));
        }

        return $data;
    }

    private function get_outbox_reply_to_bulk($outbox_ids)
    {
        $data = [];
        $res = $this->db->where_in('outbox_id', $outbox_ids)->get(db_prefix() . 'outbox_replay_to')->result_array();
        foreach ($res as $row) {
            $data[$row['outbox_id']][] = $row;
        }
        return $data;
    }

    private function get_outbox_context_bulk($outbox_ids)
    {
        $data = [];
        $res = $this->db->where_in('outbox_id', $outbox_ids)->get(db_prefix() . 'outbox_email_context')->result_array();
        foreach ($res as $row) {
            $data[$row['outbox_id']][] = $row;
        }
        return $data;
    }

    private function get_outbox_files_bulk($outbox_ids)
    {
        $data = [];
        $res = $this->db->where_in('outbox_id', $outbox_ids)->get(db_prefix() . 'outbox_files')->result_array();
        foreach ($res as $row) {
            $data[$row['outbox_id']][] = $row;
        }
        return $data;
    }


    /**
     * Get Outbox Email by ID
     * This method Get all related records in the outbox tables based on the outbox ID
    */
    public function get_outbox_email_details_by_id($outbox_id)
    {
        if (!$outbox_id) {
            return null;
        }

        $outbox_table = db_prefix() . 'outbox_email';
        $to_table     = db_prefix() . 'outbox_to';

        // Include `email_page` in the selected fields
        $this->db->select("$outbox_table.*, $to_table.id as to_id, $to_table.email as to_email, $to_table.name as to_name");
        $this->db->from($outbox_table);
        $this->db->join($to_table, "$to_table.outbox_id = $outbox_table.id", 'left');
        $this->db->where("$outbox_table.id", $outbox_id);
        $query = $this->db->get()->result_array();

        if (empty($query)) {
            return null;
        }

        // Build structured array
        $email_data = [];
        $email_data['id']          = $query[0]['id'];
        $email_data['subject']     = $query[0]['subject'];
        $email_data['message']     = $query[0]['message'];
        $email_data['created_at']  = $query[0]['created_at'];
        $email_data['email_page']  = $query[0]['email_page'];

        $email_data['recipients'] = [];

        foreach ($query as $row) {
            if (!empty($row['to_email'])) {
                $email_data['recipients'][] = [
                    'id'    => $row['to_id'],
                    'email' => $row['to_email'],
                    'name'  => $row['to_name']
                ];
            }
        }

        return $email_data;
    }
    public function get_outbox_email_details($id)
    {
        // 1. Email + subject/content
        $this->db->select('e.*, e.email_page, c.email_subject, c.email_content');
        $this->db->from('tbloutbox_email e');
        $this->db->join('tbloutbox_email_context c', 'c.outbox_id = e.id', 'left');
        $this->db->where('e.id', $id);
        $email = $this->db->get()->row_array();

        if (!$email) return null;

        // 2. All recipients
        $this->db->select('email, cc_email, bcc_email, to_type, to_user_type');
        $this->db->from('tbloutbox_to');
        $this->db->where('outbox_id', $id);
        $email['recipients'] = $this->db->get()->result_array();

        return $email;
    }


    /**
     * Delete Outbox Email by ID
     * This method deletes all related records in the outbox tables based on the outbox ID
    */
    public function delete_outbox_email_by_id($lead_id, $outbox_id)
    {

        if (!$outbox_id) {
            return false;
        }

        // Delete from tbloutbox_to
        $this->db->where('outbox_id', $outbox_id);
        $this->db->delete(db_prefix() . 'outbox_to');

        // Delete from tbloutbox_files
        $this->db->where('outbox_id', $outbox_id);
        $this->db->delete(db_prefix() . 'outbox_files');

        // Delete from tbloutbox_email_context
        $this->db->where('outbox_id', $outbox_id);
        $this->db->delete(db_prefix() . 'outbox_email_context');

        // Delete from tbloutbox_replay_to
        $this->db->where('outbox_id', $outbox_id);
        $this->db->delete(db_prefix() . 'outbox_replay_to');

        // Finally, delete the main outbox email record
        $this->db->where('id', $outbox_id);
        $this->db->delete(db_prefix() . 'outbox_email');

        return $this->db->affected_rows() > 0;
    }



}
