<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Outbox extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('Outbox_model'); // Load your custom model
        $this->load->helper('leads_helper'); // Load the leads helper for email formatting
        $this->load->helper('staff_helper');
    }

    public function index()
    {
        if (!is_admin()) {
            access_denied('Outbox');
        }

        $data['title'] = 'Outbox List';
        $data['emails'] = $this->Outbox_model->get_all_outbox();

        $this->load->view('admin/outbox/manage', $data);
    }

    public function view($id)
    {
        if (!is_admin()) {
            access_denied('View Outbox');
        }

        $data['email'] = $this->Outbox_model->get_outbox_email_details($id);
        $data['title'] = 'Outbox Details';

        $this->load->view('admin/outbox/view', $data);
    }

    // get the outbox emails for a specific lead
    public function get_outbox_page($lead_id)
    {
        $offset = (int) $this->input->post('offset');
        $limit  = (int) $this->input->post('limit');

        $records = $this->Outbox_model->get_all_outbox($lead_id, [], '', $limit, $offset, false);

        // Just load the inner HTML partial
        $this->load->view('admin/outbox/_outbox_items', ['outbox_records' => $records]);
    }

    // filter data fetch 
    public function get_filtered_outbox($lead_id)
    {
        $stages = $this->input->post('stages');
        $delivery_status = $this->input->post('delivery_status');
        $search = $this->input->post('search');

        $filters = [];

        if (!empty($stages)) {
            $filters['lead_status'] = $stages;
        }

        if (!empty($delivery_status) && $delivery_status != 'All') {
            $filters['delivery_status'] = $delivery_status;
        }

        if (!empty($search)) {
            $filters['search'] = $search;
        }

        $limit = 10;
        $offset = 0;

        $records = $this->Outbox_model->get_all_outbox($lead_id, $filters, '', $limit, $offset, false);

        $totalCount = $this->Outbox_model->get_all_outbox($lead_id, $filters, '', '', '', true);

        $html = $this->load->view('admin/outbox/_outbox_items', ['outbox_records' => $records], true);

        echo json_encode([
            'success' => true,
            'html' => $html,
            'total_pages' => ceil($totalCount / $limit)
        ]);
    }

    // get the email details for a specific outbox email
    public function get_email_detail($lead_id)
    {
        $id = $this->input->post('id');
        if (!$id) {
            echo json_encode(['success' => false, 'message' => 'Invalid Email ID.']);
            return;
        }

        $this->db->where('id', $id);
        $outbox_email_data = $this->db->get(db_prefix() . 'outbox_email')->row();

        if (!$outbox_email_data) {
            echo json_encode(['success' => false, 'message' => 'Email not found.']);
            return;
        }

        // Get lead status (name and color)
        $lead_status = $this->db->select('name, color')
            ->where('id', $outbox_email_data->lead_status)
            ->get(db_prefix() . 'leads_status')
            ->row_array();

        // Get latest email content
        $email_content = $this->db->where('outbox_id', $id)
            ->get(db_prefix() . 'outbox_email_context')
            ->row();

        if (!$email_content) {
            echo json_encode(['success' => false, 'message' => 'Email content not found.']);
            return;
        }

        // Fetch To/CC/BCC from tbloutbox_to
        $recipients = $this->db->select('email, to_type')
            ->where('outbox_id', $id)
            ->get(db_prefix() . 'outbox_to')
            ->result();

        $to = [];
        $cc = [];
        $bcc = [];

        foreach ($recipients as $r) {
            switch ($r->to_type) {
                case 'to':
                    $to[] = $r->email;
                    break;
                case 'cc':
                    $cc[] = $r->email;
                    break;
                case 'bcc':
                    $bcc[] = $r->email;
                    break;
            }
        }

        // Convert arrays to comma-separated strings
        $to_string = implode(', ', $to);
        $cc_string = implode(', ', $cc);
        $bcc_string = implode(', ', $bcc);

        $send_time = date('g:i A', strtotime($outbox_email_data->send_date));
        $time_ago = time_ago($outbox_email_data->send_date);
        $formatted_send_date = $send_time . ' (' . $time_ago . ')';

        // get the profile image of the to user and name
        $this->db->select('email,to_type,to_user_type,to_id');
        $this->db->where(['outbox_id' => $id, 'to_type' => 'to']);
        $get_to_user_data = $this->db->get(db_prefix() . 'outbox_to')->row();

        $to_user_data = get_client_staff_data_using_type_id($this, $get_to_user_data->to_id, $get_to_user_data->to_user_type);

        echo json_encode([
            'success' => true,
            'data' => [
                'email_page'  => $outbox_email_data->email_page,
                'subject'     => $email_content->email_subject,
                'content'     => $email_content->email_content,
                'lead_status' => $lead_status,
                'send_date'   => $formatted_send_date,
                'to'          => $to_string,
                'cc'          => $cc_string,
                'bcc'         => $bcc_string,
                'to_user_name' => $to_user_data['lastname'] . ', ' . $to_user_data['firstname'] . ' ' . $to_user_data['middlenaem'] ?? 'Unknown',
                'to_user_email' => $to_user_data['email'] ?? $get_to_user_data->email,
                'profile_image' => (!empty($to_user_data['profile_image']) ? base_url('uploads/staff_profile_images/' . $to_user_data['id'] . '/thumb_' . $to_user_data['profile_image']) : base_url('assets/images/user-placeholder.webp'))
            ]
        ]);
    }

    public function get_forward_outbox_email_detail($lead_id)
    {
        $id = $this->input->post('id');
        if (!$id) {
            echo json_encode(['success' => false, 'message' => 'Missing ID']);
            return;
        }

        $this->db->where('id', $id);
        $email = $this->db->get(db_prefix() . 'outbox_email')->row();

        if (!$email) {
            echo json_encode(['success' => false, 'message' => 'Email not found']);
            return;
        }

        $this->db->where('outbox_id', $id);
        $context = $this->db->get(db_prefix() . 'outbox_email_context')->row();

        $subject = $context ? $context->email_subject : '';
        $emailContent = $context ? $context->email_content : '';

        $this->db->select('email, to_type');
        $this->db->from(db_prefix() . 'outbox_to');
        $this->db->where('outbox_id', $id);
        $recipients = $this->db->get()->result();
        $to = [];
        $cc = [];
        $bcc = [];

        foreach ($recipients as $row) {
            switch (strtolower($row->to_type)) {
                case 'to':
                    $to[] = $row->email;
                    break;
                case 'cc':
                    $cc[] = $row->email;
                    break;
                case 'bcc':
                    $bcc[] = $row->email;
                    break;
            }
        }
        //  extract only what's inside <body> email
        $email_page_content =  extract_email_body($email->email_page);

        echo json_encode([
            'success' => true,
            'data' => [
                'subject' => $subject,
                'email_content' => $emailContent,
                'email_page' => $email_page_content,
                'to' => implode(', ', $to),
                'cc' => implode(', ', $cc),
                'bcc' => implode(', ', $bcc),
            ]
        ]);
    }

    // Delete an email from the outbox
    public function delete_outbox($lead_id)
    {
        if (!is_admin()) {
            access_denied('Delete Outbox');
        }

        $outbox_id = $this->input->post('id');

        if (!$outbox_id) {
            $response = [
                'success' => false,
                'message' => 'Missing outbox ID.',
            ];
            echo json_encode($response);
        }
        $Deleted_outbox_details = $this->Outbox_model->get_outbox_email_details($outbox_id);

        if ($this->Outbox_model->delete_outbox_email_by_id($lead_id, $outbox_id)) {

            // add the activity logs
            $add_activity_data = [
                'leadid' => $lead_id,
                'staffid' => get_staff_user_id(),
                'full_name' => get_staff_full_name(),
                'description' => 'Outbox Email Deleted by ' . get_staff_full_name() . ' - Subject: ' . $Deleted_outbox_details->email_content,
                'date' => date('Y-m-d H:i:s'),
            ];
            $this->db->insert(db_prefix() . 'lead_activity_log', $add_activity_data);

            set_alert('success', 'Email deleted successfully.');
            $response = [
                'success' => true,
                'message' => 'Successfully deleted outbox email.',
            ];
        } else {
            set_alert('danger', 'Failed to delete email.');
            $response = [
                'success' => false,
                'message' => 'Failed to delete outbox email.',
            ];
        }
        echo json_encode($response);
    }

    // outbox resnd email function 
    public function outbox_email_resend($lead_id)
    {
        $outbox_id = $this->input->post('id');

        if (!$outbox_id) {
            echo json_encode(['success' => false, 'message' => 'Invalid Outbox ID.']);
            return;
        }

        // Fetch email data with multiple recipients
        $outbox_email_data = $this->Outbox_model->get_outbox_email_details($outbox_id);

        if (!$outbox_email_data) {
            echo json_encode(['success' => false, 'message' => 'Outbox email not found.']);
            return;
        }

        // Collect multiple recipients
        $to_emails = [];
        $cc_emails = [];
        $bcc_emails = [];

        foreach ($outbox_email_data['recipients'] as $recipient) {
            if (!empty($recipient['to_type']) && $recipient['to_type'] === 'to' && !empty($recipient['email'])) {
                $to_emails[] = $recipient['email'];
            }
            if (!empty($recipient['to_type']) && $recipient['to_type'] === 'cc' && !empty($recipient['email'])) {
                $cc_emails[] = $recipient['email'];
            }
            if (!empty($recipient['to_type']) && $recipient['to_type'] === 'bcc' && !empty($recipient['email'])) {
                $bcc_emails[] = $recipient['email'];
            }
        }

        // Get reply-to
        $reply_to_raw = get_reply_to_emails($this, $outbox_id);
        $reply_to = [];
        foreach ($reply_to_raw as $email) {
            $reply_to[] = ['email' => $email];
        }

        // Get attachments
        $attachments = $this->db->get_where('tbloutbox_files', ['outbox_id' => $outbox_id])->result();
        $attachment_urls = [];
        foreach ($attachments as $file) {
            if (!empty($file->file_url)) {
                $attachment_urls[] = [
                    'file_name' => $file->file_name,
                    'filetype' => 'application/' . $file->file_type,
                    'file_url' => $file->file_url,
                    'download_file_url' => $file->download_file_url,
                ];
            }
        }

        // Setup email template
        $case_url = admin_url('cases/index/') . $lead_id;
        $_POST['case_url'] = '';
        $_POST['generic_email_title'] = '';
        $_POST['generic_email_back_to_case_url'] = '';
        $_POST['generic_email_content_title'] = '';
        $_POST['generic_email_content'] = extract_email_body($outbox_email_data['email_page']);
        $_POST['general_attachment'] = '';
        // $_POST['general_attachment'] = array_column($attachment_urls, 'file_url');

        $html_content = $this->load->view('admin/leads/Email_template/generic_email_template', $_POST, true);

        // Send email
        $send_result = send_mailchimp_email(
            $to_emails,
            $outbox_email_data['email_subject'],
             $_POST['generic_email_content'],
            $reply_to,
            $cc_emails,
            $bcc_emails,
            $lead_id
        );

        // ✅ Save to outbox again (resend)
        $this->Outbox_model->save_outbox_email(
            $to_emails,
            $cc_emails,
            $bcc_emails,
            $outbox_email_data['email_subject'],
            $outbox_email_data['email_content'],
            $attachment_urls,
            $outbox_email_data['lead_id'] ?? $lead_id,
            $outbox_email_data['lead_status'] ?? null,
            $_POST['generic_email_content'],
            $reply_to,
            $send_result['status'],
            $send_result['mandrill_id'] ?? null,
            1
        );

        // Log activity
        $this->db->insert(db_prefix() . 'lead_activity_log', [
            'leadid'     => $lead_id,
            'staffid'    => get_staff_user_id(),
            'full_name'  => get_staff_full_name(),
            'description' => 'Outbox Email Resent by ' . get_staff_full_name() . ' - Content: ' . $_POST['generic_email_content'],
            'date'       => date('Y-m-d H:i:s'),
        ]);

        echo json_encode(['success' => true, 'message' => 'Email resent successfully.']);
    }

    // forward outbox email sent 
    public function forward_outbox_email($lead_id)
    {
        // Retrieve POST data
        $to_raw       = $this->input->post('to');
        $cc_raw       = $this->input->post('cc');
        $bcc_raw      = $this->input->post('bcc');
        $subject      = $this->input->post('subject');
        $body         = $this->input->post('content', FALSE);
        $email_id     = $this->input->post('email_id');
        $outbox_id    = $this->input->post('outbox_id');

        // Validate required fields
        if (empty($to_raw) || empty($subject) || empty($body) || empty($lead_id)) {
            echo json_encode(['success' => false, 'message' => 'Required fields are missing.']);
            return;
        }
        // Remove Gmail proxy wrapper and restore original src
        $body = preg_replace('/https:\/\/ci\d+\.googleusercontent\.com\/[^#]+#(http[^"]+)/i', '$1', $body);

        // 2. Replace relative asset paths with full URL
        $body = str_replace(
            ['../assets/', 'http://../assets/', 'https://../assets/'],
            base_url() . 'assets/',
            $body
        );

        $outbox_lead_status = get_outbox_lead_status($this, $outbox_id);

        // Fetch group name for reply-to
        $group_name = get_the_group_name_using_leadid($this, $lead_id);

        // Prepare email template data
        $case_url = admin_url('cases/index/') . $lead_id;
        $_POST['case_url'] = '';
        $_POST['generic_email_title'] = '';
        $_POST['generic_email_back_to_case_url'] = '';
        $_POST['generic_email_content_title'] = '';
        $_POST['generic_email_content'] = $body;
        $_POST['general_attachment'] = '';

        $html_content = $this->load->view('admin/leads/Email_template/generic_email_template', $_POST, true);

        // Get reply-to email
        $reply_to_email = get_all_staff_to_replay_email($this, get_staff_user_id(), $group_name);

        // Send email using Mailchimp or SMTP logic
        $send_result = send_mailchimp_email(
            $to_raw,
            $subject,
            $html_content,
            $reply_to_email,
            $cc_raw,
            $bcc_raw,
            $lead_id
        );

        // Convert comma-separated emails to trimmed arrays
        $to  = array_map('trim', explode(',', $to_raw));
        $cc  = $cc_raw ? array_map('trim', explode(',', $cc_raw)) : [];
        $bcc = $bcc_raw ? array_map('trim', explode(',', $bcc_raw)) : [];


        // Save to outbox
        $this->Outbox_model->save_outbox_email(
            $to,
            $cc,
            $bcc,
            $subject,
            $body,
            [], // Add file attachments here if any
            $lead_id,
            $outbox_lead_status, // Optional related ID
            $html_content,
            $reply_to_email,
            $send_result['status'],
            $send_result['mandrill_id'] ?? null,
            1
        );

        // Log staff activity
        $this->db->insert(db_prefix() . 'lead_activity_log', [
            'leadid'      => $lead_id,
            'staffid'     => get_staff_user_id(),
            'full_name'   => get_staff_full_name(),
            'description' => 'Email resent by ' . get_staff_full_name() . ' - Content: ' . $body,
            'date'        => date('Y-m-d H:i:s'),
        ]);

        // Return response
        echo json_encode([
            'success'  => $send_result['status'] === 'success',
            'message'  => $send_result['status'] === 'success' ? 'Email forwarded successfully.' : 'Failed to forward email.',
            'response' => $send_result,
        ]);
    }

    // outbox delete work
    public function bulk_delete($lead_id)
    {
        $ids = $this->input->post('ids');

        if (empty($ids) || !is_array($ids)) {
            echo json_encode(['success' => false, 'message' => 'Invalid request.']);
            return;
        }

        foreach ($ids as $id) {
            // get the content before deleting outbox email
            $outbox_email_content = $this->leads_model->get_outbox_email_details($id);

            $this->db->delete(db_prefix() . 'outbox_email', ['id' => $id]);
            $this->db->delete(db_prefix() . 'outbox_email_context', ['outbox_id' => $id]);
            $this->db->delete(db_prefix() . 'outbox_to', ['outbox_id' => $id]);
            $this->db->delete(db_prefix() . 'outbox_files', ['outbox_id' => $id]);
            $this->db->delete(db_prefix() . 'outbox_replay_to', ['outbox_id' => $id]);

            // add the activity logs
            $add_activity_data = [
                'leadid' => $lead_id,
                'staffid' => get_staff_user_id(),
                'full_name' => get_staff_full_name(),
                'description' => 'Outbox Email Deleted by ' . get_staff_full_name() . ' - Content: ' .  $outbox_email_content->email_content,
                'date' => date('Y-m-d H:i:s'),
            ];
            $this->db->insert(db_prefix() . 'lead_activity_log', $add_activity_data);
        }


        echo json_encode(['success' => true]);
    }
}
